#!/usr/bin/env python3
# D11 — Light TOF (Time of Flight) — self-contained present-act engine (stdlib only)
# Control is pure boolean/ordinal; no curves, no weights, no RNG in control.
# Readouts are diagnostics only. RNG (if ever used) is ties-only (PF/Born) — not used here.

import argparse, csv, hashlib, json, math, os, sys, time
from datetime import datetime, timezone
from typing import Dict, List, Tuple

# --------------------------- Utilities ---------------------------

def utc_timestamp() -> str:
    return datetime.now(timezone.utc).strftime("%Y-%m-%dT%H-%M-%SZ")

def ensure_dirs(root: str, subdirs: List[str]) -> None:
    for d in subdirs:
        p = os.path.join(root, d)
        os.makedirs(p, exist_ok=True)

def write_text(path: str, text: str) -> None:
    with open(path, "w", encoding="utf-8") as f:
        f.write(text)

def sha256_file(path: str) -> str:
    h = hashlib.sha256()
    with open(path, "rb") as f:
        for chunk in iter(lambda: f.read(1 << 20), b""):
            h.update(chunk)
    return h.hexdigest()

def json_dump(path: str, obj: dict) -> None:
    with open(path, "w", encoding="utf-8") as f:
        json.dump(obj, f, indent=2, sort_keys=True)

# --------------------------- Core (present-act control) ---------------------------
# Geometry: square grid N×N. We define integer "shells" by the integer radius r = ⌊sqrt(dx^2+dy^2)⌋.
# A lightfront starts at radius r0 and advances one shell per tick (step_per_tick = 1 by default).
# Eligibility predicate per tick t is "cell belongs to shell r = r0 + t".
# This is pure integer logic (uses math.isqrt for floor sqrt), hence boolean/ordinal control only.

def build_shell_map(N: int, cx: int, cy: int) -> Dict[int, List[Tuple[int,int]]]:
    shells: Dict[int, List[Tuple[int,int]]] = {}
    for y in range(N):
        for x in range(N):
            dx = x - cx
            dy = y - cy
            r2 = dx*dx + dy*dy                  # integer
            r  = math.isqrt(r2)                 # floor sqrt, integer
            shells.setdefault(r, []).append((x, y))
    return shells

def simulate_lightfront(N: int, H: int, r0: int, step: int,
                        shells: Dict[int, List[Tuple[int,int]]],
                        det_radii: List[int], det_width: int) -> Dict[str, object]:
    # Arrival tick for each detector annulus (first nonzero count)
    arrival_tick = {i: None for i in range(len(det_radii))}
    per_tick_counts = []

    for t in range(H):
        r_shell = r0 + t*step                   # integer shell index this tick
        active = shells.get(r_shell, [])        # present-act eligibility
        # Diagnostics-only counts per annulus this tick
        tick_counts = [0]*len(det_radii)
        if active:
            # Count how many active cells fall into each detector annulus
            # Annulus i covers integer shells [r_i, r_i + det_width - 1]
            for i, r_i in enumerate(det_radii):
                if r_i <= r_shell <= (r_i + det_width - 1):
                    tick_counts[i] = len(active)
                    if arrival_tick[i] is None and tick_counts[i] > 0:
                        arrival_tick[i] = t
        per_tick_counts.append(tick_counts)
        # Stop early if all detectors have arrived
        if all(arrival_tick[i] is not None for i in arrival_tick):
            break

    # Build compact arrival table
    arrivals = []
    for i, r_i in enumerate(det_radii):
        arrivals.append({
            "detector_id": i,
            "radius_shells": r_i,
            "arrival_tick": arrival_tick[i]
        })
    return {"arrivals": arrivals, "per_tick_counts": per_tick_counts}

# --------------------------- Diagnostics (readout only) ---------------------------

def linear_regression_slope(xs: List[float], ys: List[float]) -> Tuple[float, float]:
    # returns slope and R^2 for y = a + b*x (b is slope wrt x)
    if len(xs) != len(ys) or len(xs) < 2:
        return float("nan"), float("nan")
    xbar = sum(xs)/len(xs); ybar = sum(ys)/len(ys)
    num = sum((x - xbar)*(y - ybar) for x, y in zip(xs, ys))
    den = sum((x - xbar)**2 for x in xs)
    if den == 0:
        return float("nan"), float("nan")
    b = num / den
    # R^2
    ss_tot = sum((y - ybar)**2 for y in ys)
    ss_res = sum((y - (ybar + b*(x - xbar)))**2 for x, y in zip(xs, ys))
    r2 = 1.0 - (ss_res/ss_tot if ss_tot != 0 else 0.0)
    return b, r2

def sr_microcheck_gammaness(c_pred: float) -> Dict[str, object]:
    # Simple SR panel: check gamma(α) closure for a few drifts (diagnostic only).
    alphas = [0.0, 0.25, 0.5, 0.75]
    rmse, preds, obs = 0.0, [], []
    for a in alphas:
        # One act: Δt=1 (units of ticks), Δx = a*c_pred*Δt; proper time Δτ^2 = Δt^2 - (Δx^2)/c^2
        dt = 1.0; dx = a*c_pred*dt; c = c_pred
        tau = math.sqrt(max(0.0, dt*dt - (dx*dx)/(c*c)))
        gamma_obs = dt / (tau if tau > 0 else 1e-12)               # avoid div by zero at a=1 (not used)
        gamma_pred = 1.0 / math.sqrt(1.0 - a*a)
        preds.append(gamma_pred); obs.append(gamma_obs)
    rmse = math.sqrt(sum((p-o)**2 for p,o in zip(preds,obs))/len(alphas))
    return {"alphas": alphas, "gamma_pred": preds, "gamma_obs": obs, "rmse": rmse, "pass": (rmse <= 1e-6)}

# --------------------------- Main run ---------------------------

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--manifest", required=True)
    ap.add_argument("--outdir", required=True)
    args = ap.parse_args()

    # Workspace
    ts = utc_timestamp()
    root = os.path.abspath(args.outdir)
    ensure_dirs(root, ["config", "outputs/metrics", "outputs/audits", "outputs/run_info", "logs"])
    # Copy manifest into config
    with open(args.manifest, "r", encoding="utf-8") as f:
        manifest = json.load(f)
    manifest_path = os.path.join(root, "config", "manifest_d11.json")
    json_dump(manifest_path, manifest)

    # Env snapshot
    env_text = []
    env_text.append(f"utc={ts}")
    env_text.append(f"os={os.name}")
    env_text.append(f"cwd={os.getcwd()}")
    env_text.append(f"python={sys.version.split()[0]}")
    write_text(os.path.join(root, "logs", "env.txt"), "\n".join(env_text))

    # Hinge/measure single-read discipline (here: we accept external hinge if provided; else fallback)
    hinge = manifest.get("hinge", {})
    Tstar = hinge.get("T_star_plus1", None)
    circ_factor = hinge.get("circ_factor", "2pi")
    R_eff = hinge.get("R_eff", None)

    # Geometry & control parameters (pure integer control)
    N = int(manifest["grid"]["N"])
    cx = int(manifest["grid"].get("cx", N//2))
    cy = int(manifest["grid"].get("cy", N//2))
    H  = int(manifest["H"])
    r0 = int(manifest["source"]["radius_shells"])
    step = int(manifest.get("step_per_tick", 1))
    det_radii = [int(r) for r in manifest["detectors"]["radii_shells"]]
    det_width = int(manifest["detectors"].get("width_shells", 1))
    dx = float(manifest["scales"].get("dx", 1.0))   # distance per shell
    dt = float(manifest["scales"].get("dt", 1.0))   # time per tick

    # Predicted c (diagnostic): prefer hinge if supplied; else fallback to step*dx/dt
    if Tstar is not None and R_eff is not None:
        Lsurf = (2.0*math.pi if str(circ_factor).lower()=="2pi" else math.pi) * float(R_eff)
        c_pred = Lsurf / float(Tstar)
        c_source = "hinge(Lsurf/T*)"
    else:
        c_pred = (step * dx) / dt
        c_source = "fallback(step*dx/dt)"

    # Build shell map once (integer sqrt), then run eligibility
    shells = build_shell_map(N, cx, cy)
    sim = simulate_lightfront(N, H, r0, step, shells, det_radii, det_width)

    # Extract arrivals (require exactly 3 by default)
    arrivals = [a for a in sim["arrivals"] if a["arrival_tick"] is not None]
    # Distances & times (diagnostics; OK to use floats here)
    distances = [(a["radius_shells"] - r0) * dx for a in arrivals]
    times     = [a["arrival_tick"] * dt for a in arrivals]

    # Regress distance vs time (distance = a + c_hat * time)
    c_hat, r2 = linear_regression_slope(times, distances)
    # Relative error and pass/fail
    rel_err = (abs(c_hat - c_pred)/c_pred) if (c_pred != 0 and not math.isnan(c_hat)) else float("nan")
    tol_rel = float(manifest["acceptance"].get("rel_err_c_max", 0.02))  # default 2%
    r2_min  = float(manifest["acceptance"].get("r2_min", 0.98))
    pass_c  = (not math.isnan(rel_err)) and (rel_err <= tol_rel) and (r2 >= r2_min)

    # SR micro-panel (diagnostic)
    sr_panel = sr_microcheck_gammaness(c_pred)
    sr_pass = bool(sr_panel["pass"])

    # Overall PASS
    passed = bool(pass_c and sr_pass)

    # --------------------------- Outputs ---------------------------
    # Metrics CSV
    metrics_csv = os.path.join(root, "outputs", "metrics", "d11_tof_arrivals.csv")
    with open(metrics_csv, "w", newline="", encoding="utf-8") as f:
        w = csv.writer(f)
        w.writerow(["detector_id", "radius_shells", "arrival_tick", "distance_units", "arrival_time_units"])
        for a, d, t in zip(arrivals, distances, times):
            w.writerow([a["detector_id"], a["radius_shells"], a["arrival_tick"], f"{d:.6f}", f"{t:.6f}"])

    # Audit JSON
    audit = {
        "sim": "D11_light_tof",
        "c_source": c_source,
        "c_pred": c_pred,
        "c_hat": c_hat,
        "r2": r2,
        "rel_error_c": rel_err,
        "tol_rel": tol_rel,
        "r2_min": r2_min,
        "pass_c": pass_c,
        "sr_panel": sr_panel,
        "pass": passed,
        "arrival_count": len(arrivals),
        "detector_spec": {
            "radii_shells": det_radii,
            "width_shells": det_width
        },
        "manifest_hash": sha256_file(manifest_path)
    }
    json_dump(os.path.join(root, "outputs", "audits", "d11_tof_audit.json"), audit)

    # Run info
    result_line = f"D11 PASS={passed} c_hat={c_hat:.6f} c_pred={c_pred:.6f} rel_err={rel_err:.6f} r2={r2:.6f}"
    write_text(os.path.join(root, "outputs", "run_info", "result_line.txt"), result_line)

    # STDOUT summary
    print(result_line)

if __name__ == "__main__":
    main()
